# Concept: `std::unique_ptr` 

`std::unique_ptr` is a smart pointer in C++ that provides exclusive ownership of a dynamically allocated object. When a `std::unique_ptr` goes out of scope, it automatically deletes the object it manages, preventing memory leaks. The key characteristics of `std::unique_ptr` are that it cannot be copied, but it can be moved, allowing ownership to be transferred.

## Key Features of `std::unique_ptr`
- Exclusive Ownership: Only one `std::unique_ptr` can own a particular resource at any time. This means no two `std::unique_ptr` objects can point to the same resource.
- Automatic Resource Management: The object managed by `std::unique_ptr` is automatically destroyed when the `unique_ptr` goes out of scope, ensuring that the resource is properly cleaned up.
- Move Semantics: `std::unique_ptr` can be moved, transferring ownership from one unique_ptr to another. This is done using `std::move`.

## Example 1: basic usage

```C++
#include <iostream>
#include <memory>

class MyClass {
public:
    MyClass() { std::cout << "MyClass Constructor" << std::endl; }
    ~MyClass() { std::cout << "MyClass Destructor" << std::endl; }
    void sayHello() const { std::cout << "Hello from MyClass" << std::endl; }
};

int main() {
    std::unique_ptr<MyClass> ptr = std::make_unique<MyClass>();
    ptr->sayHello();  // Access the object using the unique_ptr

    // No need to manually delete the object, it will be destroyed automatically
    return 0;
}
```

### Explanation:
- `std::make_unique<MyClass>()` creates a new `MyClass` object and returns a `std::unique_ptr` that manages this object.
- The object is accessed through the `unique_ptr` using the `->` operator.
- When `ptr` goes out of scope at the end of main, the destructor of `MyClass` is automatically called, and the memory is released.

### Moving std::unique_ptr
Since `std::unique_ptr` cannot be copied (to prevent multiple pointers from owning the same resource), ownership can be transferred using move semantics. This is achieved with `std::move`.

## Example 2: Transferring Ownership

```C++
#include <iostream>
#include <memory>

class MyClass {
public:
    MyClass() { std::cout << "MyClass Constructor" << std::endl; }
    ~MyClass() { std::cout << "MyClass Destructor" << std::endl; }
    void sayHello() const { std::cout << "Hello from MyClass" << std::endl; }
};

int main() {
    std::unique_ptr<MyClass> ptr1 = std::make_unique<MyClass>();

    // Transfer ownership from ptr1 to ptr2
    std::unique_ptr<MyClass> ptr2 = std::move(ptr1);

    if (!ptr1) {
        std::cout << "ptr1 is now null" << std::endl;
    }

    ptr2->sayHello();  // Access the object through ptr2

    return 0;
}
```

### Explanation:

- `std::move(ptr1)` transfers ownership of the object from `ptr1` to `ptr2`.
- After the move, `ptr1` is set to `nullptr,` and attempting to dereference it would be unsafe.
- `ptr2` now owns the object and can be used to access it.

## Example 3: Custom Deleters 

`std::unique_ptr` allows you to specify a custom deleter, which is a function or function object that is called when the `unique_ptr` goes out of scope. This is useful when managing resources that require special cleanup, such as files or network sockets.

```C++
#include <iostream>
#include <memory>
#include <cstdio>  // For FILE*

struct FileDeleter {
    void operator()(FILE* fp) const {
        if (fp) {
            std::cout << "Closing file" << std::endl;
            std::fclose(fp);
        }
    }
};

int main() {
    std::unique_ptr<FILE, FileDeleter> filePtr(std::fopen("example.txt", "w"));

    if (filePtr) {
        std::fprintf(filePtr.get(), "Hello, file!\n");
    }

    // filePtr goes out of scope, and FileDeleter is invoked to close the file

    return 0;
}
```

### Explanation:

- `FileDeleter` is a custom deleter that closes the file when the unique_ptr goes out of scope.
- `std::unique_ptr<FILE, FileDeleter> filePtr(std::fopen("example.txt", "w"));` creates a `unique_ptr` to manage a `FILE*` resource.
- When `filePtr` goes out of scope, the custom deleter is called, which closes the file.

## Common Use Cases for std::unique_ptr

- Resource Management: Use `std::unique_ptr` to manage dynamically allocated resources like memory, file handles, or network sockets, ensuring they are automatically released when no longer needed.
- Move-Only Objects: `std::unique_ptr` is ideal for managing resources in classes or functions that need to transfer ownership, such as factory functions or objects that manage resources with exclusive ownership.
- Custom Cleanup: When dealing with resources that require special cleanup actions, `std::unique_ptr` with a custom deleter provides a clean and safe way to manage those resources.

## Summary

- `std::unique_ptr` is a smart pointer that provides exclusive ownership of a dynamically allocated object.
- Automatic Resource Management: The object is automatically deleted when the unique_ptr goes out of scope, preventing memory leaks.
- Move Semantics: `std::unique_ptr` can be moved but not copied, allowing safe transfer of ownership.
- Custom Deleters: `std::unique_ptr` supports custom deleters for handling special cleanup requirements.

`std::unique_ptr` is a fundamental tool in modern C++ for ensuring safe and efficient resource management, making it easier to avoid memory leaks and dangling pointers.

