# Command Line Arguments

Command line arguments in C++ provide a way to pass information to a program when it is executed from the command line. This feature allows for greater flexibility and interactivity in program execution, enabling users to customize program behavior without modifying the source code.

## Key Characteristics

- Command line arguments are passed to the `main()` function
- The `main()` function typically takes two parameters: `argc` (argument count) and `argv` (argument vector)
- `argc` is an integer representing the number of arguments (including the program name)
- `argv` is an array of C-style strings (char*) containing the actual arguments
- The first argument (`argv`) is always the name of the program itself
- Arguments are separated by spaces in the command line


## Example 1: Basic Usage: Displaying Command Line Arguments

```cpp
#include <iostream>

int main(int argc, char* argv[]) {
    std::cout << "Number of arguments: " << argc << std::endl;
    
    for (int i = 0; i < argc; ++i) {
        std::cout << "Argument " << i << ": " << argv[i] << std::endl;
    }
    
    return 0;
}
```

### Explanation:

- This program simply prints out all the command line arguments it receives.
- `argc` gives the count of arguments, including the program name.
- `argv` is an array of C-style strings containing each argument.
- We iterate through `argv` to print each argument.

## Example 2: Parsing Numeric Arguments: Calculator

```cpp
#include <iostream>
#include <string>
#include <cstdlib>

int main(int argc, char* argv[]) {
    if (argc != 4) {
        std::cerr << "Usage: " << argv[0] << " <number1> <operation> <number2>" << std::endl;
        return 1;
    }

    double num1 = std::atof(argv[1]);
    std::string op = argv[2];
    double num2 = std::atof(argv[3]);
    double result;

    if (op == "+") result = num1 + num2;
    else if (op == "-") result = num1 - num2;
    else if (op == "*") result = num1 * num2;
    else if (op == "/") {
        if (num2 == 0) {
            std::cerr << "Error: Division by zero" << std::endl;
            return 1;
        }
        result = num1 / num2;
    }
    else {
        std::cerr << "Error: Invalid operation" << std::endl;
        return 1;
    }

    std::cout << "Result: " << result << std::endl;
    return 0;
}
```

### Explanation:

- This program implements a simple calculator using command line arguments.
- It checks if the correct number of arguments (4) is provided.
- `std::atof()` is used to convert string arguments to floating-point numbers.
- The program performs the specified operation and prints the result.
- Error handling is included for invalid input and division by zero.

## Example 3: Handling Flags: File Processing Program

```cpp
#include <iostream>
#include <string>
#include <algorithm>

int main(int argc, char* argv[]) {
    bool verbose = false;
    std::string inputFile;
    std::string outputFile;

    for (int i = 1; i < argc; ++i) {
        std::string arg = argv[i];
        if (arg == "-v" || arg == "--verbose") {
            verbose = true;
        } else if (arg == "-i" || arg == "--input") {
            if (i + 1 < argc) {
                inputFile = argv[++i];
            } else {
                std::cerr << "Error: Input file name missing" << std::endl;
                return 1;
            }
        } else if (arg == "-o" || arg == "--output") {
            if (i + 1 < argc) {
                outputFile = argv[++i];
            } else {
                std::cerr << "Error: Output file name missing" << std::endl;
                return 1;
            }
        } else {
            std::cerr << "Unknown argument: " << arg << std::endl;
            return 1;
        }
    }

    if (inputFile.empty() || outputFile.empty()) {
        std::cerr << "Usage: " << argv[0] << " -i <input_file> -o <output_file> [-v]" << std::endl;
        return 1;
    }

    if (verbose) {
        std::cout << "Input file: " << inputFile << std::endl;
        std::cout << "Output file: " << outputFile << std::endl;
    }

    // File processing logic would go here

    std::cout << "File processing complete." << std::endl;
    return 0;
}
```

### Explanation:

- This example demonstrates how to handle command line flags and options.
- It processes arguments to set a verbose flag and specify input/output files.
- The program checks for the presence of required arguments and handles errors.
- It shows how to handle both short (-v) and long (--verbose) versions of flags.

## Example 4: Using Command Line Arguments with Standard Library Containers

```cpp
#include <iostream>
#include <vector>
#include <algorithm>
#include <string>

int main(int argc, char* argv[]) {
    if (argc < 2) {
        std::cerr << "Usage: " << argv[0] << " <word1> [word2] [word3] ..." << std::endl;
        return 1;
    }

    std::vector<std::string> words(argv + 1, argv + argc);

    std::cout << "Original words:" << std::endl;
    for (const auto& word : words) {
        std::cout << word << " ";
    }
    std::cout << std::endl;

    std::sort(words.begin(), words.end());

    std::cout << "Sorted words:" << std::endl;
    for (const auto& word : words) {
        std::cout << word << " ";
    }
    std::cout << std::endl;

    return 0;
}
```

### Explanation:

- This example shows how to use command line arguments with C++ standard library containers.
- We create a vector of strings from the command line arguments, skipping the program name.
- The program prints the original list of words, then sorts and prints the sorted list.
- This demonstrates how easily command line arguments can be integrated with C++ algorithms and containers.

## Additional Considerations

- Argument Validation: Always validate the number and type of arguments to ensure your program receives the expected input.
- Error Handling: Provide clear error messages when incorrect arguments are supplied.
- Security: Be cautious when using command line arguments directly in system calls or SQL queries to prevent injection attacks.
- Cross-platform Considerations: On Windows, arguments are typically passed as Unicode strings, which may require special handling.

## Summary

Command line arguments in C++ provide a powerful mechanism for making programs more flexible and interactive. By utilizing the `argc` and `argv` parameters in the `main()` function, developers can create programs that accept various inputs and flags from the command line.

Key points to remember:
1. The `main()` function receives arguments through `int argc` and `char* argv[]`.
2. `argc` provides the count of arguments, while `argv` contains the actual argument strings.
3. The first argument (`argv`) is always the program name.
4. Arguments can be parsed and converted to appropriate data types as needed.
5. Flags and options can be implemented to control program behavior.
6. Standard C++ containers and algorithms can be easily used with command line arguments.

By effectively using command line arguments, C++ programmers can create more versatile and user-friendly applications that can be easily integrated into scripts and command-line workflows.

## Related
- How do I pass command line arguments in C++
- What are the common pitfalls when using command line arguments in C++
- How can I validate command line arguments in C++
- What libraries are available for parsing command line arguments in C++
- How do I handle optional command line arguments in C++
